<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       http://michael.lapkin.net/bandaid
 * @since      1.0.0
 * @package    Band_Aid
 * @subpackage Band_Aid/public
 * @author     Michael Lapkin <mslapkin@gmail.com>
 */
class Band_Aid_Public {

	/**
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $band_aid    The ID of this plugin.
	 */
	private $band_aid;

	/**
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * @since    1.0.0
	 * @param      string    $band_aid       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $band_aid, $version ) {

		$this->band_aid = $band_aid;
		$this->version = $version;
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		wp_enqueue_style( $this->band_aid, plugin_dir_url( __FILE__ ) . 'css/band-aid-public.css', array(), $this->version, 'all' );

	}
/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		wp_enqueue_script( $this->band_aid, plugin_dir_url( __FILE__ ) . 'js/band-aid-public.js', array( 'jquery' ), $this->version, false );

	}

	/**
	 * Register the shortcode.
	 *
	 * @since    1.0.0
	 */

	public function init_shortcode(){
		add_shortcode( 'bandaid', array( $this, 'proc_shortcode' ) );
	}

	private function show_tables(){
		/**
		 * retrieve table names from db; return html
		 *
		 * @since    1.0.0
		 * @return   string                             HTML text

		 */
		global $wpdb;
		$tables=$wpdb->get_results("SHOW TABLES");
		
		ob_start();

		// get table names
		// pull in html with php insertions of variable data
		include( plugin_dir_path( __FILE__ ) . 'partials/band-aid-public-table-list.php' );

		echo "<hr>";

		$html = ob_get_contents();
		ob_end_clean();

		return $html;
	}


	private function show_cols($table){

		/**
		 * retrieve field names from db; return html
		 *
		 * @since    1.0.0
		 * @param    string               $table        Name of table
		 * @return   string                             HTML text

		 */
		global $wpdb;

		// display list of valid number fields 
		//$sqlstr = "Select column_name from information_schema.columns where table_name='" . $table . "'";	
		$sql = $wpdb->prepare( "Select column_name from information_schema.columns where table_name=%s", $table );
		$res = $wpdb->get_results( $sql);	
		ob_start();

		// pull in html with php insertions of variable data
		include( plugin_dir_path( __FILE__ ) . 'partials/band-aid-public-col-list.php' );

		$html = ob_get_contents();
		ob_end_clean();

		return $html;
	}
	private function is_valid($table, $column = null){

		/**
		 * checks for table
		 *
		 * @since    1.0.0
		 * @param    string               $table        Name of table
		 * @return   string                             HTML text

		 */
		global $wpdb;

		// display list of valid number fields 
		$sqlstr = "SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME=%s" . (isset($column) ? " AND COLUMN_NAME=%s" : "");
		$sql = isset($column) ? $wpdb->prepare($sqlstr, $table, $column) : $wpdb->prepare($sqlstr, $table );
		$res = $wpdb->get_results($sql);
		return !empty($res);
	}



	public function proc_shortcode( $atts = [], $content = null, $tag = '' ) {
		/**
		 * Process the shortcode callback (render plugin)
		 * @since    1.0.0
		 */

		// normalize attribute keys, lowercase
		$atts = array_change_key_case( (array) $atts, CASE_LOWER );

		$err=array();
		$NUM_COLORS=6;
		$TITLE_COLOR=0;
		$GOAL_COLOR=1;
		$BAR1_COLOR=2;
		$BAR2_COLOR=3;
		$TOTAL_COLOR=4;
		$COUNT_COLOR=5;
		$a_colors = Array("rgb(40,40,40)","rgb(20, 108, 195)","rgb(20, 108, 195)","rgb(119, 195, 236)","rgb(20, 108, 195)","rgb(40,40,40)");

		// validate goal
		if (isset($atts['goal']) && is_numeric($atts['goal'])){
			$goal = $atts['goal'];
		} else {
			array_push($err,array( "'goal' not specified. ","* Specifiy a 'goal' attribute in the shortcode.  ex: '<b>goal='5000'</b><br>"));
		}


	// check for supplied configurations for progress bar (current or table/column)
		if(isset($atts['current'])) {
			//  static config configured
			if (is_numeric($atts['current'])) {
				// VALID "static" CONFIG (goal/current)
				$current = $atts['current'];
			} else {
				array_push($err,array( "'current' is not a numeric field.","* Specifiy a 'current' attribute in the shortcode (unformatted).  ex: '<b>current='5000'</b><br>"));
			}
			
		} elseif (isset($atts['table']) && isset($atts['column'])){
			// VALID active config
			$table = $atts['table'];
			$col = $atts['column'];
			
			// test table 
			if ($this->is_valid($table)){
				// test table  & col
				if (!$this->is_valid($table, $col)){
					array_push($err,array( $col . " is not a valid column.",$this->show_cols($table)));
				}

			} else {
				array_push($err,array( $table . " is not a valid table.",$this->show_tables()));
			}
		} else {
			array_push($err,array("'current' or 'table/column' not specified.",$this->show_tables()));
		}

		// pickup optional attributes 
		$title_text = esc_html((isset($atts['title_text']))? $atts['title_text'] : "");
		$show_count = isset($atts['show_count']) && ($atts['show_count'] != "") ;
		$goal_text = (isset($atts['goal_text']) && is_string($atts['goal_text'])) ? $atts['goal_text'] : "";
		$count_text = (isset($atts['count_text']) && is_string($atts['count_text'])) ? $atts['count_text'] : "donations";
		$current_offset = (isset($atts['current_offset']) && is_numeric($atts['current_offset'])) ? $atts['current_offset'] : 0;
		$count = (isset($atts['count']) && is_numeric($atts['count'])) ? $atts['count'] : 0;
		$count_offset = (isset($atts['count_offset']) && is_numeric($atts['count_offset'])) ? $atts['count_offset']: 0;

		// overwrite colors
		if (isset($atts['colors'])){
			$a_colors_temp = explode("|", $atts['colors']);
			if (count($a_colors_temp) == $NUM_COLORS){
				// valid color string 
				$a_colors = $a_colors_temp;
			} else {
				array_push($err, array("color configuration has " . count($a_colors_temp) . " colors.  (6 are required).", "* review color scheme"));

			}
		}
	
		// set color vals 
		$title_color = esc_html($a_colors[$TITLE_COLOR]);
		$goal_color = esc_html($a_colors[$GOAL_COLOR]);
		$bar_color_1 = esc_html($a_colors[$BAR1_COLOR]);
		$bar_color_2 = esc_html($a_colors[$BAR2_COLOR]);
		$total_color = esc_html($a_colors[$TOTAL_COLOR]);
		$counter_color = esc_html($a_colors[$COUNT_COLOR]);
		
		if ($err){
			echo "<br>Bandaid Configuration<br>";
			// display errors
			foreach ($err as $e) {
				echo $e[0] . "<br>";
			}
			echo "<hr>";
			// display additional data
			foreach ($err as $e) {
				echo $e[1] . "<br>";
			}
		} else {

			global $wpdb;
			$html="";
			$countstr="";

			// if ACTIVE config (current was not set)
			if (!isset($current)){
				// get current from database
				$wpdb->hide_errors();

				// run query 
				$sqlstr = "SELECT SUM(" . $col . ") 'current', COUNT(" . $col . ") AS count FROM " . $table;
				$res = $wpdb->get_row($sqlstr);

				if($wpdb->last_error == '') {
					// adjust result & set $current value
					$current = $res->current -  $current_offset;
					$count = $res->count - $count_offset;
				} else {
					// db error 
					echo "Bandaid configuration: check configured table/column values.";
					$wpdb->print_error();
					unset($current); 
					unset($count); 

				}
			}

			// Display results (current was either supplied or returned from the database)
			if (isset($current) && is_numeric($current)){
				if ($show_count){
				// build count string 
					$countstr = str_replace("%c", $count, $count_text, $found);
					if (!$found){
						$countstr = $count . " " . $count_text;
					}
					$countstr = esc_html(str_replace("%d", "$" . number_format($current), $countstr, $found));
				}
				
				// set goal string 
				$goalstr = str_replace("%d", "$" . number_format($goal), $goal_text, $found);
				if (!$found){
					$goalstr = $goal_text . " $" . number_format($goal) ;
				}
				$goalstr = esc_html($goalstr);

				$per = $current/$goal;	// percentage of goal reached
				$per_lim = max(0, min(1, $per));  // limited percent for display 

				// height of progress bar
				$height = round($per_lim * 245);	// 245 is progress bar height - padding
				
				// set top of current value;
				$top = round(225 - ($per_lim * 225)) ; // 225 is vertical range of current value display

				// format total 
				$current = esc_html(number_format($current));

				ob_start();

				// pull in html with php insertions of variable data
				include( plugin_dir_path( __FILE__ ) . 'partials/band-aid-public-display.php' );

				$html = ob_get_contents();

				ob_end_clean();
			}


			
			// return output
			return $html;
		}
	}

}
